<?php

namespace App\Controllers;

class DatabaseController extends BaseController
{
    protected $db;

    public function __construct()
    {
        $this->db = \Config\Database::connect();
    }

    public function index()
    {
        $setting = $this->appSetting();
        $data = [

            'setting' => $setting,
            'title' => 'Database Aplikasi',
        ];
        return view('Panel/Database/data_view', $data);
    }

    public function exportJson()
    {
        // Ambil nama tabel dari parameter GET, misalnya ?tables=peserta,kelas,jurusan
        $tables = $this->request->getGet('tables');

        if (!$tables) {
            return $this->response->setStatusCode(400)->setJSON([
                'status' => false,
                'message' => 'Parameter "tables" tidak ditemukan.'
            ]);
        }

        // Ubah string menjadi array
        $tableList = explode(',', $tables);

        // Tambahkan tabel 'settings' jika belum ada
        if (!in_array('settings', $tableList)) {
            $tableList[] = 'settings';
        }

        $data = [];

        foreach ($tableList as $table) {
            try {
                $query = $this->db->table($table)->get();
                $data[$table] = $query->getResultArray();
            } catch (\Throwable $e) {
                // Skip tabel yang error (misal typo atau tidak ada)
                $data[$table] = ['error' => $e->getMessage()];
            }
        }

        $jsonOutput = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        $filename = 'backup_' . date('Ymd_His') . '.json';

        return $this->response
            ->setHeader('Content-Type', 'application/json')
            ->setHeader('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->setBody($jsonOutput);
    }

    public function restoreJson()
    {
        $file = $this->request->getFile('backup_json');

        if (!$file || !$file->isValid()) {
            return $this->response->setJSON(['status' => false, 'message' => 'File tidak valid'])->setStatusCode(400);
        }

        $content = file_get_contents($file->getTempName());
        $json = json_decode($content, true);

        if (!$json || !is_array($json)) {
            return $this->response->setJSON(['status' => false, 'message' => 'Format file JSON tidak valid'])->setStatusCode(400);
        }

        try {
            $this->db->disableForeignKeyChecks();

            foreach ($json as $table => $rows) {
                // Kosongkan tabel dulu
                $this->db->table($table)->truncate();

                if (!empty($rows)) {
                    $this->db->table($table)->insertBatch($rows);
                }
            }

            $this->db->enableForeignKeyChecks();

            return $this->response->setJSON(['status' => true, 'message' => 'Data berhasil direstore dari JSON.']);
        } catch (\Throwable $e) {
            $this->db->enableForeignKeyChecks();
            return $this->response->setJSON([
                'status' => false,
                'message' => 'Restore gagal: ' . $e->getMessage()
            ])->setStatusCode(500);
        }
    }

    public function deleteTables()
    {
        $input = $this->request->getJSON(true);
        $tables = $input['tables'] ?? [];

        if (empty($tables)) {
            return $this->response->setJSON([
                'status' => false,
                'message' => 'Tidak ada tabel yang dipilih.'
            ]);
        }

        try {
            // Nonaktifkan foreign key checks sementara
            $this->db->disableForeignKeyChecks();

            // Urutkan agar tabel anak dihapus dulu
            $priorityOrder = [
                'jawaban',
                'hasil_ujian',
                'soal_opsi',
                'soal',
                'topik_soal',
                'banksoal',
                'ujian',
                'peserta',
                'kelas',
                'jurusan',
                'tingkat',
                'agama',
            ];

            // Urutkan tabel berdasarkan prioritas
            usort($tables, function ($a, $b) use ($priorityOrder) {
                $aIndex = array_search($a, $priorityOrder);
                $bIndex = array_search($b, $priorityOrder);
                return ($aIndex !== false ? $aIndex : PHP_INT_MAX) - ($bIndex !== false ? $bIndex : PHP_INT_MAX);
            });

            // Truncate tiap tabel
            foreach ($tables as $table) {
                $this->db->table($table)->truncate();
            }

            $this->db->enableForeignKeyChecks();

            return $this->response->setJSON([
                'status' => true,
                'message' => 'Data berhasil dihapus dari tabel terpilih.'
            ]);
        } catch (\Throwable $e) {
            $this->db->enableForeignKeyChecks(); // pastikan tetap diaktifkan walau error
            return $this->response->setJSON([
                'status' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage(),
            ]);
        }
    }
}
