<?php

namespace App\Controllers;

use Ramsey\Uuid\Uuid;
use App\Controllers\BaseController;
use App\Models\BankSoalModel;
use App\Models\UserModel;

class BankSoalController extends BaseController
{
    protected $bankSoalModel;
    protected $validation;

    public function __construct()
    {
        $this->bankSoalModel = new BankSoalModel();
        $this->validation    = \Config\Services::validation();
    }

    public function index()
    {

        $userModel = new UserModel();
        if (has_role('admin')) {
            $gurus = $userModel->where('roles', 'guru')->findAll();
        } elseif (has_role('guru')) {
            $userId = user_id();
            $gurus = [$userModel->find($userId)];
        } else {
            // Role tidak dikenali, tolak akses
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound();
        }

        $data = [
            'title'      => 'Bank Soal',
            'gurus'      => $gurus,

            'setting'    => $this->appSetting(),
        ];

        return view('Panel/BankSoal/bank_soal_view', $data);
    }


    public function getAll()
    {
        if (!$this->request->isAJAX()) {
            return $this->response->setStatusCode(403)->setJSON([
                'status'  => false,
                'message' => 'Hanya bisa diakses via AJAX.'
            ]);
        }

        if (!is_logged_in()) {
            return $this->response->setStatusCode(401)->setJSON([
                'status'  => false,
                'message' => 'Silakan login terlebih dahulu.'
            ]);
        }

        // Ambil data berdasarkan role
        if (has_role('admin')) {
            $data = $this->bankSoalModel->getAll(); // semua data
        } else {
            $data = $this->bankSoalModel->getAll(user_id()); // data milik guru
        }

        return $this->response->setJSON([
            'status' => true,
            'data'   => $data
        ]);
    }


    public function create()
    {
        if ($this->request->isAJAX()) {
            $rules = [
                'kode'       => 'required|min_length[2]|max_length[50]|is_unique[bank_soal.kode]',
                'nama'       => 'required|min_length[3]|max_length[255]',
                'deskripsi'  => 'permit_empty',
                'is_active'  => 'in_list[0,1]',
                'created_by' => 'required|min_length[36]|max_length[36]',
            ];

            if (!$this->validate($rules)) {
                return $this->response->setJSON([
                    'status'  => false,
                    'message' => $this->validation->getErrors()
                ]);
            }

            $data = [
                'id'         => Uuid::uuid4()->toString(),
                'kode'       => $this->request->getPost('kode'),
                'nama'       => $this->request->getPost('nama'),
                'deskripsi'  => $this->request->getPost('deskripsi'),
                'is_active'  => $this->request->getPost('is_active') ?? 1,
                'is_public'  => $this->request->getPost('is_public') ?? 1,
                'created_by' => $this->request->getPost('created_by'),
                'created_at' => date('Y-m-d H:i:s'),
            ];

            $this->bankSoalModel->insert($data);

            return $this->response->setJSON([
                'status'  => true,
                'message' => 'Bank soal berhasil ditambahkan.'
            ]);
        }

        return $this->fail('Hanya bisa diakses via AJAX.');
    }

    public function update($id = null)
    {
        if ($this->request->isAJAX()) {
            $rules = [
                'kode'       => "required|min_length[2]|max_length[50]|is_unique[bank_soal.kode,id,{$id}]",
                'nama'       => 'required|min_length[3]|max_length[255]',
                'deskripsi'  => 'permit_empty',
                'is_active'  => 'in_list[0,1]'
            ];

            if (!$this->validate($rules)) {
                return $this->response->setJSON([
                    'status'  => false,
                    'message' => $this->validation->getErrors()
                ]);
            }

            $data = [
                'kode'       => $this->request->getPost('kode'),
                'nama'       => $this->request->getPost('nama'),
                'deskripsi'  => $this->request->getPost('deskripsi'),
                'is_active'  => $this->request->getPost('is_active') ?? 1,
                'is_public'  => $this->request->getPost('is_public') ?? 1,
                'updated_at' => date('Y-m-d H:i:s'),
            ];

            $this->bankSoalModel->update($id, $data);

            return $this->response->setJSON([
                'status'  => true,
                'message' => 'Bank soal berhasil diperbarui.'
            ]);
        }

        return $this->fail('Hanya bisa diakses via AJAX.');
    }
    public function update_visibility($id = null)
    {
        if ($this->request->isAJAX()) {
            $is_public = $this->request->getPost('is_public');

            // Validasi
            if (!in_array($is_public, ['0', '1'], true)) {
                return $this->response->setJSON([
                    'status'  => false,
                    'message' => 'Nilai visibilitas tidak valid.'
                ]);
            }

            // Update field is_public
            $this->bankSoalModel->update($id, [
                'is_public'  => $is_public,
                'updated_at' => date('Y-m-d H:i:s')
            ]);

            return $this->response->setJSON([
                'status'  => true,
                'message' => 'Visibilitas bank soal berhasil diperbarui.'
            ]);
        }

        return $this->fail('Hanya bisa diakses via AJAX.');
    }


    public function delete($id = null)
    {
        if ($this->request->isAJAX()) {
            // Cek apakah sedang dipakai di ujian
            $db = \Config\Database::connect();
            $used = $db->table('ujian')->where('bank_soal_id', $id)->countAllResults();

            if ($used > 0) {
                return $this->response->setJSON([
                    'status'  => false,
                    'message' => 'Bank soal tidak dapat dihapus karena sedang digunakan.'
                ]);
            }

            $this->bankSoalModel->delete($id);
            return $this->response->setJSON([
                'status'  => true,
                'message' => 'Bank soal berhasil dihapus.'
            ]);
        }

        return $this->fail('Hanya bisa diakses via AJAX.');
    }

    private function fail($message)
    {
        return $this->response->setJSON([
            'status'  => false,
            'message' => $message
        ]);
    }
}
